﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Resources;
using QualityManagement.MediatR;
using QualityManagement.MediatR.RiskManagement;

namespace QualityManagement.API.Controllers.Risk_Management
{
    /// <summary>
    /// Controller for Risk
    /// </summary>
    [Route("api/[controller]")]
    [ApiController]
    [Authorize]
    public class RiskController(
        IMediator _mediator) : BaseController
    {
        /// <summary>
        /// Create  A risk
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        [ClaimCheck("ADD_RISK_MANAGEMENT")]
        public async Task<IActionResult> CreateRisk(AddRiskCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Delete Risk by Risk Id
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpDelete("{id}")]
        [ClaimCheck("DELETE_RISK_MANAGEMENT")]
        public async Task<IActionResult> DeleteRisk(Guid id)
        {
            var result = await _mediator.Send(new DeleteRiskCommand() { Id = id });
            return GenerateResponse(result);
        }

        /// <summary>
        /// Upadate Risk by Risk Id
        /// </summary>
        /// <param name="id"></param>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPut("{id}")]
        [ClaimCheck("EDIT_RISK_MANAGEMENT")]
        public async Task<IActionResult> UpdateRisk(Guid id, UpdateRiskCommand command)
        {
            command.Id = id;
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Get Risk by Risk Id
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpGet("{id}")]
        public async Task<IActionResult> GetRisk(Guid id)
        {
            var result = await _mediator.Send(new GetRiskCommand() { Id = id });
            return GenerateResponse(result);
        }

        /// <summary>
        /// get all Risk
        /// </summary>
        /// <param name="riskResource"></param>
        /// <returns></returns>
        [HttpGet]
        [ClaimCheck("VIEW_RISK_MANAGEMENT")]
        public async Task<IActionResult> GetCapaRequests([FromQuery] RiskResource riskResource)
        {
            var getRiskListCommand = new GetRiskListCommand()
            {
                RiskResource = riskResource
            };
            var result = await _mediator.Send(getRiskListCommand);
            var paginationMetadata = new
            {
                totalCount = result.TotalCount,
                pageSize = result.PageSize,
                skip = result.Skip,
                totalPages = result.TotalPages
            };
            Response.Headers.Append("X-Pagination",
                Newtonsoft.Json.JsonConvert.SerializeObject(paginationMetadata));

            return Ok(result);
        }

        /// <summary>
        /// get all Asign Risk 
        /// </summary>
        /// <returns></returns>
        [HttpGet("assigned")]
        public async Task<IActionResult> GetAllAsignRisk()
        {
            var result = await _mediator.Send(new GetAllAsignRiskCommand());
            return GenerateResponse(result);
        }
    }
}
